/**
 *  @file    DefaultVariables
 *  @author  moca
 *  @version 1.0.3 2023/09/29
 */

/*:ja
 * @plugindesc v1.0.3 ゲーム起動時に変数に値を予め設定します
 * @author moca
 * @help ゲーム起動時に変数に値を予め設定します。デバッグ実行時も有効です。
 * 
 * ※数値を文字列として登録することはできません。
 * また、一度設定したパラメータは変更を保存しないと適用されません。
 * 
 * ## バージョン履歴
 * 2023/09/29 1.0.3 ニューゲーム選択時に値を設定できる設定を追加
 * 2023/09/26 1.0.2 スイッチに対応
 * 2023/09/26 1.0.1 設定値の型制限を排除。数値を文字列として登録できない旨を追記。
 * 2023/09/26 1.0.0 初版
 * 
 * 利用規約：
 * プラグイン作者に無断で使用、改変、再配布は不可です。 
 * 
 * 
 * @param VariableSettings
 * @text 変数設定
 * @desc 変数の初期値を設定します
 * @type struct<VariableData>[]
 * 
 * @param SwitchSettings
 * @text スイッチ設定
 * @desc ONにするスイッチ番号を設定します
 * @type switch[]
 * 
 * @param DisableNewGame
 * @text ニューゲーム時に無効化
 * @desc ニューゲーム選択時に初期値を設定するかを指定します。trueで無効化。
 * @type boolean
 * @default false
*/

/*~struct~VariableData:
 * @param id
 * @text 変数番号
 * @desc 変数番号を指定します
 * @type variable
 * 
 * @param value
 * @text 設定値
 * @desc 変数番号に設定する値を指定します
 */

(function() {
    'use strict';

    // プラグインパラメータ
    const PLUGIN_NAME = document.currentScript.src.split("/").pop().replace(/\.js$/, "");
    const PARAMS = PluginManager.parameters(PLUGIN_NAME);
    const DEFAULT_VARIABLES = PARAMS["VariableSettings"] === '' ? null : JSON.parse(PARAMS["VariableSettings"]);
    const DEFAULT_SWITCHES = PARAMS["SwitchSettings"] === '' ? null : JSON.parse(PARAMS["SwitchSettings"]);
    const DISABLE_NEWGAME = PARAMS["DisableNewGame"] === 'true' ? true : false;

    // 変数のデフォルト値を設定する
    function setDefaultVariables() {
        if(DEFAULT_VARIABLES) {
            DEFAULT_VARIABLES.forEach(e => {
                var param = JSON.parse(e);
                $gameVariables.setValue(param.id, param.value);
            });
        }
    };

    // スイッチのデフォルト値を設定する
    function setDefaultSwitches() {
        if(DEFAULT_SWITCHES) {
            DEFAULT_SWITCHES.forEach(e => {
                $gameSwitches.setValue(e, true);
            });
        }
    };

    var _Scene_Boot_start      = Scene_Boot.prototype.start;
    Scene_Boot.prototype.start = function() {
        _Scene_Boot_start.apply(this, arguments);

        setDefaultVariables();
        setDefaultSwitches();
    };

    var _DataManager_setupNewGame = DataManager.setupNewGame;
    DataManager.setupNewGame = function() {
        _DataManager_setupNewGame.apply(this, arguments);

        if(DISABLE_NEWGAME) {
            return;
        }
        setDefaultVariables();
        setDefaultSwitches();
    };
})();